﻿using System.Globalization;
using System.Text;
using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.StaticFiles;
using Microsoft.Extensions.FileProviders;

namespace MiniApiStaticFileDemo
{
    public class SortableDirectoryFormatter : IDirectoryFormatter
    {
        private readonly string _pageTitle;
        // 时间格式：年-月-日 时:分:秒（UTC 时区）
        private const string CustomDateTimeFormat = "yyyy-MM-dd HH:mm:ss +00:00";

        // 构造函数：支持自定义页面标题
        public SortableDirectoryFormatter(string pageTitle = "目录列表")
        {
            _pageTitle = pageTitle ?? "目录列表";
        }

        public async Task GenerateContentAsync(HttpContext context, IEnumerable<IFileInfo> contents)
        {
            // 1. 过滤隐藏文件 + 初始排序（目录在前，文件在后）
            var safeContents = contents ?? Enumerable.Empty<IFileInfo>();
            var fileList = new List<IFileInfo>(safeContents.Where(f =>
                !string.IsNullOrEmpty(f.Name) && !f.Name.StartsWith(".")
            ));
            fileList.Sort((a, b) =>
            {
                if (a.IsDirectory != b.IsDirectory)
                    return a.IsDirectory ? -1 : 1;
                return string.Compare(a.Name, b.Name, StringComparison.OrdinalIgnoreCase);
            });

            // 2. 生成面包屑导航（支持多层目录跳转）
            var breadcrumbBuilder = new StringBuilder();
            string currentPath = "/";
            breadcrumbBuilder.Append($"<a href=\"{HtmlEncode(currentPath)}\">/</a>");
            var requestPath = context.Request.Path.Value;
            if (!string.IsNullOrEmpty(requestPath))
            {
                var pathSegments = requestPath.Split('/', StringSplitOptions.RemoveEmptyEntries | StringSplitOptions.TrimEntries);
                foreach (var segment in pathSegments)
                {
                    currentPath = Path.Combine(currentPath, segment + "/").Replace(Path.DirectorySeparatorChar, '/');
                    breadcrumbBuilder.Append($"<a href=\"{HtmlEncode(currentPath)}\">{HtmlEncode(segment)}/</a>");
                }
            }

            // 3. 构建 HTML 页面（含样式 + 排序脚本）
            var htmlBuilder = new StringBuilder(4096);
            htmlBuilder.AppendLine("<!DOCTYPE html>");
            htmlBuilder.AppendLine("<html lang=\"zh-CN\">");
            htmlBuilder.AppendLine("<head>");
            htmlBuilder.AppendLine($"<title>{_pageTitle} - {HtmlEncode(requestPath ?? "/")}</title>");
            // 样式：保持简洁，适配不同设备
            htmlBuilder.AppendLine(@"<style>
            body { font-family: ""Segoe UI"", ""Microsoft YaHei"", sans-serif; font-size: 14px; max-width: 1200px; margin: 0 auto; padding: 20px; }
            header h1 { font-size: 24px; font-weight: 400; margin: 0 0 20px 0; color: #333; }
            #index { width: 100%; border-collapse: separate; border-spacing: 0; border: 1px solid #eee; }
            #index th { background: #f8f9fa; padding: 10px; text-align: center; cursor: pointer; user-select: none; position: relative; border-bottom: 2px solid #ddd; }
            #index td { padding: 8px 10px; border-bottom: 1px solid #eee; }
            #index th:hover { background: #f1f3f5; }
            #index td.length, td.modified { text-align: right; }
            a { color: #127aac; text-decoration: none; }
            a:hover { color: #13709e; text-decoration: underline; }
            .sort-arrow { position: absolute; right: 8px; font-size: 0.8em; color: #999; }
            .dir-name { font-weight: 500; }
        </style>");
            htmlBuilder.AppendLine("</head>");
            htmlBuilder.AppendLine("<body>");
            htmlBuilder.AppendLine($"<header><h1>{_pageTitle}：{breadcrumbBuilder}</h1></header>");
            htmlBuilder.AppendLine("<table id=\"index\">");
            htmlBuilder.AppendLine("<thead><tr>");
            htmlBuilder.AppendLine("<th data-col=\"name\">名称 <span class=\"sort-arrow\">↑</span></th>");
            htmlBuilder.AppendLine("<th data-col=\"size\">大小 <span class=\"sort-arrow\"></span></th>");
            htmlBuilder.AppendLine("<th data-col=\"modified\">最后修改时间 <span class=\"sort-arrow\"></span></th>");
            htmlBuilder.AppendLine("</tr></thead><tbody>");

            // 4. 生成文件/目录列表行
            if (fileList.Count == 0)
            {
                htmlBuilder.AppendLine("<tr><td colspan=\"3\" style=\"text-align:center; padding:20px; color:#666;\">目录无可用文件</td></tr>");
            }
            else
            {
                foreach (var file in fileList)
                {
                    var fileName = file.IsDirectory ? $"{file.Name}/" : file.Name;
                    var fileClass = file.IsDirectory ? "dir-name" : "";
                    var fileSize = file.IsDirectory ? "-" : FormatFileSizeWithComma(file.Length);
                    var fileModified = file.LastModified.ToUniversalTime().ToString(CustomDateTimeFormat, CultureInfo.InvariantCulture);
                    var encodedFileName = Uri.EscapeDataString(file.Name);
                    var fileUrl = $"{context.Request.Path}/{encodedFileName}".Replace("//", "/");

                    htmlBuilder.AppendLine("<tr>");
                    htmlBuilder.AppendLine($"<td class=\"name {fileClass}\"><a href=\"{HtmlEncode(fileUrl)}\">{HtmlEncode(fileName)}</a></td>");
                    htmlBuilder.AppendLine($"<td class=\"length\">{HtmlEncode(fileSize)}</td>");
                    htmlBuilder.AppendLine($"<td class=\"modified\">{HtmlEncode(fileModified)}</td>");
                    htmlBuilder.AppendLine("</tr>");
                }
            }

            htmlBuilder.AppendLine("</tbody></table>");
            // 5. 排序核心脚本（点击表头切换排序）
            htmlBuilder.AppendLine(@"<script>
            document.addEventListener('DOMContentLoaded', function() {
                const table = document.getElementById('index');
                if (!table) return;
                const headers = table.querySelectorAll('thead th[data-col]');
                const tbody = table.querySelector('tbody');
                let currentSort = { col: 'name', order: 'asc' };

                // 表头点击事件
                headers.forEach(th => {
                    th.addEventListener('click', function() {
                        const newCol = this.dataset.col;
                        currentSort.order = (currentSort.col === newCol) ? (currentSort.order === 'asc' ? 'desc' : 'asc') : 'asc';
                        currentSort.col = newCol;
                        updateSortArrows();
                        sortTable();
                    });
                });

                // 更新排序箭头
                function updateSortArrows() {
                    headers.forEach(th => {
                        const arrow = th.querySelector('.sort-arrow');
                        arrow.textContent = (th.dataset.col === currentSort.col) ? (currentSort.order === 'asc' ? '↑' : '↓') : '';
                    });
                }

                // 排序逻辑
                function sortTable() {
                    const rows = Array.from(tbody.querySelectorAll('tr'));
                    if (!rows.length) return;

                    rows.sort((a, b) => {
                        const cellA = a.querySelector(`td.${getCellClass(currentSort.col)}`).textContent.trim();
                        const cellB = b.querySelector(`td.${getCellClass(currentSort.col)}`).textContent.trim();

                        switch (currentSort.col) {
                            case 'name':
                                const isDirA = cellA.endsWith('/');
                                const isDirB = cellB.endsWith('/');
                                if (isDirA !== isDirB) return isDirA ? -1 : 1;
                                return currentSort.order === 'asc' ? cellA.localeCompare(cellB, 'zh-CN') : cellB.localeCompare(cellA, 'zh-CN');
                            case 'size':
                                if (cellA === '-') return -1;
                                if (cellB === '-') return 1;
                                const sizeA = BigInt(cellA.replace(/,/g, ''));
                                const sizeB = BigInt(cellB.replace(/,/g, ''));
                                return currentSort.order === 'asc' ? (sizeA < sizeB ? -1 : 1) : (sizeA > sizeB ? -1 : 1);
                            case 'modified':
                                const dateA = new Date(cellA.replace(' +00:00', 'Z')).getTime();
                                const dateB = new Date(cellB.replace(' +00:00', 'Z')).getTime();
                                return currentSort.order === 'asc' ? (dateA - dateB) : (dateB - dateA);
                            default: return 0;
                        }
                    });

                    tbody.innerHTML = '';
                    rows.forEach(row => tbody.appendChild(row));
                }

                function getCellClass(colName) {
                    return colName === 'name' ? 'name' : colName === 'size' ? 'length' : 'modified';
                }

                updateSortArrows();
            });
        </script>");
            htmlBuilder.AppendLine("</body></html>");

            // 输出响应
            context.Response.ContentType = "text/html; charset=utf-8";
            await context.Response.WriteAsync(htmlBuilder.ToString(), Encoding.UTF8);
        }

        // 辅助方法：文件大小格式化（带千分位）
        private static string FormatFileSizeWithComma(long bytes)
        {
            return bytes.ToString("N0", CultureInfo.InvariantCulture);
        }

        // 辅助方法：HTML 编码（防 XSS）
        private static string HtmlEncode(string value)
        {
            if (string.IsNullOrEmpty(value)) return string.Empty;
            return value.Replace("&", "&amp;")
                        .Replace("<", "&lt;")
                        .Replace(">", "&gt;")
                        .Replace("\"", "&quot;")
                        .Replace("'", "&#39;")
                        .Replace("`", "&#96;");
        }
    }
}